/*	MotivePower.cpp - Created by Giampiero Caprino

This file is part of Train Director 3

Train Director is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; using exclusively version 2.
It is expressly forbidden the use of higher versions of the GNU
General Public License.

Train Director is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Train Director; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

#include "wx/wx.h"
#include "MotivePower.h"
#include "Options.h"

Array<MotivePower *> gMotivePowerCache;
MotivePower *gEditorMotivePower;

IntOption       editor_gauge(wxT("editor_gauge"),
                             wxT("Default Track Gauge"),
                             wxT("Editor"), 0);

const Char *power_clean(const Char *p)
{
        static Char clean[128];
        int max = sizeof(clean) / sizeof(clean[0]);
        int x = 0;

        for(x = 0; *p; ++p) {
            if(*p == ' ')
                continue;
            if(*p == '\n')
                break;
            if(x < max - 1)
                clean[x++] = *p;
        }
        clean[x] = 0;
        clean[max - 1] = 0;
        return clean;
}

static MotivePower *power_find(MotivePower *p)
{
        int cnt;
        for(int i = 0; i < gMotivePowerCache.Length(); ++i) {
            MotivePower *pc = gMotivePowerCache.At(i);
            if(pc->_power.Length() != p->_power.Length())
                continue;
            // make sure all components in pc are present in p
            for(cnt = pc->_power.Length(); --cnt >= 0; ) {
                if(!p->Contains(pc->_power.At(cnt)))
                    break;
            }
            if(cnt < 0)
                return pc;
        }
        return 0;
}

MotivePower *power_parse(const Char *p)
{
        MotivePower tmp;
        tmp.Add(p); // parse and split into components
//        p = power_clean(p);
        MotivePower *pc = power_find(&tmp);
        if(pc)
            return pc;
        pc = new MotivePower();
        pc->Add(p);
        int i;
        for(i = gMotivePowerCache.Length(); --i >= 0; ) {
            MotivePower *cached = gMotivePowerCache.At(i);
            if(cached->SameAs(pc)) {
                return pc;
            }
        }
        gMotivePowerCache.Add(pc);
        return pc;
}

void    power_select(const Char *pwr)
{
        pwr = power_clean(pwr);
        if(!*pwr) {
            gEditorMotivePower = 0;
            return;
        }
        wxString str(pwr); // need to copy it, since it's a static in power_clean
        MotivePower tmp;
        tmp.Add(str.c_str()); // parse and split into components
        MotivePower *pc = power_find(&tmp);
        if(!pc) {
            pc = power_parse(str.c_str());
        }
        gEditorMotivePower = pc;
}

//
//
//

MotivePower::MotivePower()
{
}


MotivePower::~MotivePower()
{
        Clear(); // release strings in _power array
}

void    MotivePower::Clear()
{
        _power.Release();
}

void    MotivePower::Add(const Char *pwr)
{
        Char *dup = wxStrdup(pwr);
        Char *p, *start;
        const Char *p1;

        start = dup;
        for(p = start; *p; ) {
            if(*p != ',') {
                ++p;
                continue;
            }
            if(p != start) { // handle , at the start of the string
                *p++ = 0;
                p1 = power_clean(start);
                if(*p1)
                    _power.Add(wxStrdup(p1));
                start = p;
            } else
                start = ++p; // skip , record new string start
        }
        if(p != start) {
            p1 = power_clean(start);
            if(*p1)
                _power.Add(wxStrdup(p1));
        }
        free(dup);
}

bool    MotivePower::Contains(MotivePower *other)
{
        const Char *p1;
        int i, j;
        Array<const Char *>& pwr = other->_power;

        for(i = pwr.Length(); --i >= 0; ) {
            p1 = pwr.At(i);
            for(j = _power.Length(); --j >= 0; ) {
                if(wxStrcmp(_power.At(j), p1) == 0)
                    return true;
            }
        }
        return false;
}

bool    MotivePower::Contains(const Char *pwr)
{
        int j;
        for(j = _power.Length(); --j >= 0; ) {
            if(wxStrcmp(_power.At(j), pwr) == 0)
                return true;
        }
        return false;
}

bool    MotivePower::SameAs(MotivePower *other)
{
        const Char *p1;
        int i, j;
        int found = 0;
        Array<const Char *>& pwr = other->_power;

        for(i = pwr.Length(); --i >= 0; ) {
            p1 = pwr.At(i);
            for(j = _power.Length(); --j >= 0; ) {
                if(wxStrcmp(_power.At(j), p1) == 0)
                    ++found;
            }
        }
        return found == _power.Length(); // all our elements are also present in 'other'
}

void    MotivePower::ToString(wxString& out)
{
        int     i;
        const Char *sep = wxT("");

        out.Clear();
        for(i = 0; i < _power.Length(); ++i) {
            out.append(sep);
            out.append(_power.At(i));
            sep = wxT(",");
        }
}
