#include "IconSelectionDialog.h"
#include "wx/dir.h"
#include "wx/filename.h"
#include "wx/image.h"

IconSelectionDialog *gIconSelectionDialog = 0;

enum {
    ID_ICON_BASE = 2000
};

BEGIN_EVENT_TABLE(IconSelectionDialog, wxDialog)
    EVT_BUTTON(wxID_OK, IconSelectionDialog::OnOK)
    EVT_BUTTON(wxID_CANCEL, IconSelectionDialog::OnCancel)
END_EVENT_TABLE()

IconSelectionDialog::IconSelectionDialog(wxWindow *parent)
    : wxDialog(parent, wxID_ANY, wxT("Select Icon"), wxDefaultPosition, wxSize(600, 400),
               wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER, wxT("Icon Selection"))
    , m_selectedIndex(-1)
{
    wxBoxSizer* mainSizer = new wxBoxSizer(wxVERTICAL);
    
    // Create header text
    wxStaticText* header = new wxStaticText(this, wxID_ANY, wxT("Select an icon for the layout:"));
    mainSizer->Add(header, 0, wxALL, 10);
    
    // Create scrolled window for icons
    m_scrolledWindow = new wxScrolledWindow(this, wxID_ANY, wxDefaultPosition, wxSize(580, 300));
    m_scrolledWindow->SetScrollRate(10, 10);
    m_scrolledWindow->SetBackgroundColour(*wxWHITE);
    
    // Create flex grid sizer for icons (4 columns)
    m_iconSizer = new wxFlexGridSizer(0, 4, 10, 10);
    m_iconSizer->SetFlexibleDirection(wxBOTH);
    
    m_scrolledWindow->SetSizer(m_iconSizer);
    mainSizer->Add(m_scrolledWindow, 1, wxEXPAND | wxALL, 5);
    
    // Create button sizer
    wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);
    m_okButton = new wxButton(this, wxID_OK, wxT("OK"));
    m_okButton->Enable(false); // Disabled until an icon is selected
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, wxT("Cancel"));
    
    buttonSizer->Add(m_okButton, 0, wxRIGHT, 5);
    buttonSizer->Add(cancelButton, 0, 0);
    
    mainSizer->Add(buttonSizer, 0, wxALIGN_RIGHT | wxALL, 10);
    
    SetSizer(mainSizer);
    
    // Load available icons
    LoadIcons();
    
    Centre();
}

IconSelectionDialog::~IconSelectionDialog()
{
}

void IconSelectionDialog::LoadIcons()
{
    // Look for .xpm files in common directories
    wxArrayString searchPaths;
    
    // Add current working directory
    searchPaths.Add(wxGetCwd());
    
    // Add common icon directories
    searchPaths.Add(wxGetCwd() + wxT("/icons"));
    searchPaths.Add(wxGetCwd() + wxT("/images"));
    searchPaths.Add(wxGetCwd() + wxT("/pixmaps"));
    
    // Search for .xpm files
    for (size_t i = 0; i < searchPaths.GetCount(); ++i) {
        wxString path = searchPaths[i];
        if (!wxDirExists(path))
            continue;
            
        wxDir dir(path);
        if (!dir.IsOpened())
            continue;
            
        wxString filename;
        bool cont = dir.GetFirst(&filename, wxT("*.xpm"), wxDIR_FILES);
        while (cont) {
            wxString fullPath = path + wxFileName::GetPathSeparator() + filename;
            
            // Try to load the image
            wxImage image;
            if (image.LoadFile(fullPath, wxBITMAP_TYPE_XPM)) {
                // Scale image if too large (max 48x48 for display)
                if (image.GetWidth() > 48 || image.GetHeight() > 48) {
                    image = image.Scale(48, 48, wxIMAGE_QUALITY_HIGH);
                }
                
                IconInfo *iconInfoPtr = new IconInfo();
                IconInfo& iconInfo = *iconInfoPtr;
                iconInfo.path = fullPath;
                iconInfo.bitmap = wxBitmap(image);
                
                // Create panel for this icon
                iconInfo.panel = new wxPanel(m_scrolledWindow, ID_ICON_BASE + m_icons.Length());
                iconInfo.panel->SetBackgroundColour(*wxWHITE);
                
                wxBoxSizer* iconPanelSizer = new wxBoxSizer(wxVERTICAL);
                
                // Create bitmap control
                iconInfo.control = new wxStaticBitmap(iconInfo.panel, wxID_ANY, iconInfo.bitmap);
                iconPanelSizer->Add(iconInfo.control, 0, wxALIGN_CENTER | wxALL, 2);
                
                // Create label with filename
                wxString label = wxFileName(filename).GetName(); // Remove extension
                if (label.Length() > 12) {
                    label = label.Left(9) + wxT("...");
                }
                iconInfo.label = new wxStaticText(iconInfo.panel, wxID_ANY, label);
                iconInfo.label->SetFont(wxFont(8, wxFONTFAMILY_DEFAULT, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL));
                iconPanelSizer->Add(iconInfo.label, 0, wxALIGN_CENTER | wxALL, 2);
                
                iconInfo.panel->SetSizer(iconPanelSizer);
                iconInfo.panel->SetMinSize(wxSize(60, 80));
                
                // Bind click event
                iconInfo.panel->Connect(iconInfo.panel->GetId(), wxEVT_LEFT_DOWN, 
                       wxMouseEventHandler(IconSelectionDialog::OnIconSelected));
                iconInfo.control->Connect(iconInfo.control->GetId(), wxEVT_LEFT_DOWN, 
                       wxMouseEventHandler(IconSelectionDialog::OnIconSelected));
                iconInfo.label->Connect(iconInfo.label->GetId(), wxEVT_LEFT_DOWN, 
                       wxMouseEventHandler(IconSelectionDialog::OnIconSelected));
//                iconInfo.control->Bind(wxEVT_LEFT_DOWN, &IconSelectionDialog::OnIconSelected, this);
//              iconInfo.label->Bind(wxEVT_LEFT_DOWN, &IconSelectionDialog::OnIconSelected, this);
                
                m_icons.Add(iconInfoPtr);
                m_iconSizer->Add(iconInfo.panel, 0, wxALL, 2);
            }

            cont = dir.GetNext(&filename);
        }
    }
    
    // If no icons found, show a message
    if (m_icons.Length() == 0) {
        wxStaticText* noIconsText = new wxStaticText(m_scrolledWindow, wxID_ANY, 
            wxT("No .xpm icon files found.\nPlace .xpm files in the current directory,\nor in 'icons', 'images', or 'pixmaps' subdirectories."));
        noIconsText->SetForegroundColour(*wxRED);
        m_iconSizer->Add(noIconsText, 0, wxALL, 10);
    }

    m_scrolledWindow->FitInside();
}

void IconSelectionDialog::OnIconSelected(wxMouseEvent& event)
{
    wxObject* eventObject = event.GetEventObject();
    Array<IconInfo *> *m_icons = &gIconSelectionDialog->m_icons;

    // Find which icon was clicked
    for (size_t i = 0; i < m_icons->Length(); ++i) {
        IconInfo *icon = m_icons->At(i);
        if (eventObject == icon->panel || eventObject == icon->control || eventObject == icon->label) {
            
            // Deselect previous selection
            if (gIconSelectionDialog->m_selectedIndex >= 0 && gIconSelectionDialog->m_selectedIndex < (int)m_icons->Length()) {
                IconInfo *oldIcon = m_icons->At(gIconSelectionDialog->m_selectedIndex);
                oldIcon->panel->SetBackgroundColour(*wxWHITE);
                oldIcon->panel->Refresh();
            }
            
            // Select new icon
            gIconSelectionDialog->m_selectedIndex = i;
            gIconSelectionDialog->m_selectedPath = icon->path;
            icon->panel->SetBackgroundColour(wxColor(200, 200, 255)); // Light blue
            icon->panel->Refresh();
            
            // Enable OK button
            gIconSelectionDialog->m_okButton->Enable(true);
            break;
        }
    }
}

void IconSelectionDialog::OnOK(wxCommandEvent& event)
{
    if (m_selectedIndex >= 0) {
        EndModal(wxID_OK);
    }
}

void IconSelectionDialog::OnCancel(wxCommandEvent& event)
{
    EndModal(wxID_CANCEL);
}

int IconSelectionDialog::ShowModal()
{
    return wxDialog::ShowModal();
}